#! /usr/local/bin/R
# -----------------------------------------------------------


# ---------------------------------------------------------------------
message("Log file for code executed at\n")
message(format(Sys.time(), "%a %b %d %X %Y"))
# ---------------------------------------------------------------------


# ---------------------------------------------------------------------
library(magrittr)
library(glue);
library(lubridate)
library(stringr)
library(haven);
library(roll)
library(data.table);
library(statar)
# ---------------------------------------------------------------------


# ---------------------------------------------------------------------
# --- 1. Import and add ISO Country names to currencies:
dt_fx <- fread("../task_data/output/FX_daily_long.csv")
dt_fx <- dt_fx[, .(date= ymd(date), ticker, base, foreign, prc=as.numeric(value))]

# Merge with predicted base factors from PCs
dt_pca <- fread("./output/base_PCA.csv") 
dt_fx <- merge(dt_fx, dt_pca, by = c("date", "base"))

dt_pca_2 <- dt_pca[, foreign := base]
dt_pca_2 <- dt_pca_2[, -c("base", "year")]
dt_fx <- merge(dt_fx, dt_pca_2, by = c("date", "foreign"), all.x = T)

dt_ccodes <- read_dta("./input/country-codes.dta") %>% data.table

dt_fx <- merge(dt_fx, dt_ccodes[, .(base=ISO4217, base_country=ISO3166)], by = c("base"), all.x=T)
dt_fx <- merge(dt_fx, dt_ccodes[, .(foreign=ISO4217, foreign_country=ISO3166)], by = c("foreign"), all.x=T)
dt_fx[base=="TWD",base_country:="TWN"]
dt_fx[foreign=="TWD",foreign_country:="TWN"]
dt_fx[base=="HKD",base_country:="HKG"]
dt_fx[foreign=="HKD",foreign_country:="HKG"]
dt_fx[base=="EUR",base_country:="EUR"]
dt_fx[foreign=="EUR",foreign_country:="EUR"]

#drop weekends on some tickers
dt_fx <- dt_fx[wday(date) %in% c(2,3,4,5,6) ,] 

# Merge with predicted vol from PCs
dt_vol <- fread("./output/volatility_PCA.csv") 
dt_fx <- merge(dt_fx, dt_vol, by = c("date", "base_country", "foreign_country"))

dt_fx[]
# ---------------------------------------------------------------------


# ---------------------------------------------------------------------
# --- 2. Constructing Base Factors (as in Lustig/Richmond(2018)), but here

setorder(dt_fx, ticker, date)
dt_fx[, datey := year(date) ]
dt_fx[, log_prc := log(prc)]
dt_fx[, d1_prc  := d1_prc_syst] 

dt_fx[, range_euro := 0 ]
for (c_euro in c("ATS", "BEF", "FIM", "FRF", "DEM", "GRD", 
                 "IEP", "ITL", "NLG", "PTE", "ESP")){
  dt_fx[ base == c_euro | foreign == c_euro, range_euro := 1 ]
}

## some statistics:

# Fix pre euro currencies
dt_fx <- dt_fx[ datey > 1999 & range_euro == 1 , prc := NA ]
dt_fx <- dt_fx[ datey > 1999 & range_euro == 1 , log_prc := NA ]
dt_fx <- dt_fx[ datey > 1999 & range_euro == 1 , d1_prc := NA ]

setnames(dt_fx, "resid.x", "d1_base_factor")
setnames(dt_fx, "resid.y", "d1_foreign_factor")

dt_fx <- dt_fx[ !is.na(base_country)]
dt_fx <- dt_fx[ !is.na(foreign_country)]

# From now on remove euro from analysis as in LR */
dt_fx <- dt_fx[ base != "EUR"]
dt_fx <- dt_fx[ foreign != "EUR"]
# ---------------------------------------------------------------------


# ---------------------------------------------------------------------
# --- 3. ROLLING REGRESSIONS
dt_fx_reg <- copy(dt_fx)

# STD DEVIATION OF FACTORS
dt_single <- dt_fx_reg[, .(date, base_country, d1_base_factor) ] %>% unique
setorder(dt_single, base_country, date)
dt_single[, sd_d1_base  := roll_sd(d1_base_factor,  width=1800L, min_obs=100L), 
          by = .(base_country) ]

dt_single[is.nan(sd_d1_base), sd_d1_base_pre := NA]
dt_single[]

# COVARIANCE OF FACTORS and Std. Dev of LHS
setorder(dt_fx_reg, base_country, foreign_country, date)
dt_fx_reg[, cov_base_foreign  := roll_cov(d1_prc, d1_base_factor, width=1800L, min_obs=100L),
          by = .(base_country, foreign_country)]

dt_fx_reg[, sd_d1_prc := roll_sd(d1_prc, width=1800L, min_obs=100L), by = .(base_country, foreign_country)]	

dt_fx_reg <- dt_fx_reg[ datey > 1999 & range_euro == 1 , sd_d1_prc := NA ] # Don't calculate for pre-euro currencies after 1990!
dt_fx_reg <- dt_fx_reg[ is.na(d1_prc) , sd_d1_prc := NA ] # Don't calculate if current FX is missing!

# Std. dev of difference in base and foreign averages
dt_fx_reg[, sd_d1_base_foreign  := roll_sd(d1_base_factor-d1_foreign_factor, width=1800L, min_obs=100L), 
          by = .(base_country, foreign_country)] 

# Covariance  
dt_fx_reg[, cov_bases := roll_cov(d1_base_factor, d1_foreign_factor, width=1800L, min_obs=100L),
          by = .(base_country, foreign_country)] ## Covariance of base factors


dt_fx_reg[is.nan(cov_base_foreign), cov_base_foreign := NA]
dt_fx_reg[is.nan(sd_d1_prc), sd_d1_prc := NA]
dt_fx_reg[]

# Now merge it back
rm(dt_fx)
dt_fx_reg <- merge(dt_fx_reg, dt_single[, .(date, base_country, sd_d1_base)], 
                   all.x = T, by = c("date", "base_country"))
dt_fx_reg <- merge(dt_fx_reg, dt_single[, .(date, foreign_country=base_country, sd_d1_foreign=sd_d1_base)], 
                   all.x = T, by = c("date", "foreign_country"))
dt_fx_reg[]


dt_fx_reg <- dt_fx_reg[ datey > 1999 & range_euro == 1 , sd_d1_base := NA ] # Don't calculate for pre-euro currencies after 1999!
dt_fx_reg <- dt_fx_reg[ datey > 1999 & range_euro == 1 , cov_base_foreign := NA ] # Don't calculate for pre-euro currencies after 1999!
dt_fx_reg <- dt_fx_reg[ is.na(d1_prc) , sd_d1_base := NA ] # Don't calculate if current FX is missing!
dt_fx_reg <- dt_fx_reg[ is.na(d1_prc) , cov_base_foreign := NA ] # Don't calculate if current FX is missing!


# we can estimate betas and R2 directly from covariances and variances
dt_fx_reg[, beta  :=  cov_base_foreign  / sd_d1_base^2 ]
dt_fx_reg[, r2   :=  cov_base_foreign^2  / (sd_d1_base * sd_d1_prc)^2 ]

# new measure I invented
dt_fx_reg[, var_adj  := sd_d1_prc^2 / (sd_d1_base^2 + sd_d1_foreign^2) ]

# MEASURE AS IN COCHRANE LONGSTAFF SANTA CLARA
dt_fx_reg[, unshared_cls1  := sd_d1_prc^2 / sd_d1_base_foreign^2 ]
# Different way of getting at it...
dt_fx_reg[, unshared_cls2  := sd_d1_prc^2 / (sd_d1_base^2 + sd_d1_foreign^2 - 2*cov_bases) ]

dt_fx_reg[]
# ---------------------------------------------------------------------


# ---------------------------------------------------------------------
## Export
dt_export <- dt_fx_reg[, .(date, base, foreign, base_country, foreign_country, d1_prc, log_prc, fx_vol = sd_d1_prc,
                           beta, R2_y = r2, var_adj, sd_d1_base, cov_base_foreign,
                           unshared_cls1, unshared_cls2)]
dt_export[, datey := year(date) ]

dt_export <- dt_export[, lapply(.SD, function(x) mean(x, na.rm=T)), 
                       .SDcols = c("d1_prc", "log_prc", "fx_vol", "beta", "R2_y", "var_adj", "unshared_cls1", "unshared_cls2", "sd_d1_base", "cov_base_foreign"),
                       by=.(datey, base, foreign, base_country, foreign_country)]
dt_export[]


write_dta(dt_export, "./output/betas_annuals_PCA_resid.dta")
# ---------------------------------------------------------------------

